/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var _1 = require("./");
var path = require("path");
;
var ErrorInformation = (function () {
    function ErrorInformation(options) {
        if (!options.rawError) {
            throw new Error("You must provide an error to process");
        }
        this._rawError = options.rawError;
        this._errorType = options.errorType;
        this._packageInfo = options.packageInfo;
        this._exeName = defaultValue(path.basename(process.execPath), options.exeName);
        this._errorReporter = options.errorReporter;
        this._pathMode = defaultValue(_1.pathMode.defaultMode, options.mode);
    }
    ErrorInformation.prototype.getStructuredStack = function () {
        if (!this._structuredError) {
            try {
                this._structuredError = this.rawToStructured(this._rawError.stack);
            }
            catch (e) {
                if (this.errorReporter) {
                    this.errorReporter.emitInternalError(e, this._rawError, "");
                }
                throw e;
            }
        }
        return this._structuredError;
    };
    ErrorInformation.prototype.getStandardStack = function () {
        if (!this._standardStack) {
            try {
                this._standardStack = _1.structuredToStandard(this.getStructuredStack());
            }
            catch (e) {
                if (this.errorReporter) {
                    this.errorReporter.emitInternalError(e, this._rawError, "");
                }
                throw e;
            }
        }
        return this._standardStack;
    };
    ErrorInformation.prototype.getVerboseStack = function () {
        if (!this._verboseStack) {
            try {
                this._verboseStack = _1.structuredToSemiWatson(this.getStructuredStack());
            }
            catch (e) {
                if (this.errorReporter) {
                    this.errorReporter.emitInternalError(e, this._rawError, "");
                }
                throw e;
            }
        }
        return this._verboseStack;
    };
    Object.defineProperty(ErrorInformation.prototype, "errorName", {
        get: function () {
            return this._rawError.name;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "errorMessage", {
        get: function () {
            return this._rawError.message;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "errorType", {
        get: function () {
            return this._errorType;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "rawError", {
        get: function () {
            return this._rawError;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "packageInfo", {
        get: function () {
            return this._packageInfo;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "exeName", {
        get: function () {
            return this._exeName;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorInformation.prototype, "errorReporter", {
        get: function () {
            return this._errorReporter;
        },
        enumerable: true,
        configurable: true
    });
    ErrorInformation.prototype.rawToStructured = function (raw) {
        var stackProc = new _1.StackProcessor({
            appRoot: this.packageInfo.packageJsonFullPath,
            errorReporter: this.errorReporter,
            packageInfo: this.packageInfo,
            mode: this._pathMode });
        var originalLines = raw.split("\n");
        var structuredStack;
        try {
            structuredStack = stackProc.parseFirstLine(originalLines[0]);
        }
        catch (e) {
            structuredStack = { ErrorType: this.errorName, ErrorMessage: this.errorMessage, Stack: [] };
        }
        for (var i = 1; i < originalLines.length; i++) {
            var currentFrame = void 0;
            try {
                currentFrame = stackProc.parseStackLine(originalLines[i], true);
            }
            catch (e) {
                if (this.errorReporter) {
                    this.errorReporter.emitInternalError(e, this.rawError, originalLines[i]);
                }
                currentFrame = {
                    RelativePath: "UNSUPPORTED",
                    extra: {
                        failedToParse: true,
                        rawString: originalLines[i]
                    }
                };
            }
            structuredStack.Stack.push(currentFrame);
        }
        return structuredStack;
    };
    return ErrorInformation;
}());
exports.ErrorInformation = ErrorInformation;
function setStackTraceLimit(limit) {
    try {
        Error.stackTraceLimit(limit);
    }
    catch (e) {
        var err = Error;
        err.stackTraceLimit = limit;
    }
}
exports.setStackTraceLimit = setStackTraceLimit;
function coerceToError(e) {
    if (e instanceof Error) {
        return e;
    }
    return new NonError(e);
}
exports.coerceToError = coerceToError;
var NonError = (function (_super) {
    __extends(NonError, _super);
    function NonError(input) {
        _super.call(this);
        this.name = "NonError";
        if (input.message) {
            this.message = input.message;
        }
        else {
            this.message = JSON.stringify(input);
        }
        if (input.stack) {
            this.stack = input.stack;
        }
        else {
            var header = this.name + ": " + this.message;
            this.stack = header + "\nat NonError";
        }
        this.originalObject = input;
    }
    return NonError;
}(Error));
exports.NonError = NonError;
function defaultValue(defaultValue, actual, strict) {
    if (strict === void 0) { strict = true; }
    if ((strict && actual === undefined) || (!strict && !actual)) {
        return defaultValue;
    }
    return actual;
}
exports.defaultValue = defaultValue;

// SIG // Begin signature block
// SIG // MIIh8wYJKoZIhvcNAQcCoIIh5DCCIeACAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // 9oRvSx4Ngr4RIeC7W/yfKOSIaOcHvVi4PqeBzwVNJveg
// SIG // gguEMIIFDDCCA/SgAwIBAgITMwAAAU/nxmLJRvSpfwAA
// SIG // AAABTzANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTE2MTExNzIxNTkxNFoX
// SIG // DTE4MDIxNzIxNTkxNFowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBALSJkIp2DKzFPaG8gejXUAi7
// SIG // WmmnrsfMuFu9OA8hUUYII9G9Iplztyjv9werK/lI96UU
// SIG // BalNJBttYZacg50zwCx5mfBd8+zVRN+lzVn6dq0GP8ZK
// SIG // A+bqLZQQwmZv/WLQUIrRIpEjl2U0ocSaAl6S4rh8KHjC
// SIG // XuNG98rvOFeIrFYGWMqYG8eo1142rppFMOO2tz0Ywqcn
// SIG // sHlNU9gy5najUCr2P7q7QR4khk2Ym3qe9izivJBgZfuW
// SIG // afrgE2jiiSoudRidwm24IfSZXip/MaCnDczvTxFu/klC
// SIG // TmHeN8srnkDvzqtecAhzpXJ7t0bOWzJTXYo5UW+w16h4
// SIG // 3aMob17uQh8CAwEAAaOCAXswggF3MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjc9BgEGCCsGAQUFBwMDMB0GA1UdDgQWBBSS
// SIG // DqERcbYvVQ3DD/TJXHQuWgxqvzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwODY1
// SIG // K2I0YjEyODc4LWUyOTMtNDNlOS1iMjFlLTdkMzA3MTlk
// SIG // NDUyZjAfBgNVHSMEGDAWgBTm/F97uyIAWORyTrX0IXQj
// SIG // MubvrDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNDb2RTaWdQQ0FfMjAxMC0wNy0wNi5jcmwwWgYIKwYB
// SIG // BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNp
// SIG // Z1BDQV8yMDEwLTA3LTA2LmNydDAMBgNVHRMBAf8EAjAA
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQARwLlqQzX+hbE4gA24
// SIG // dHUsjXFJuDxB7S294gY2kHYrieW2gemG4xiMlpMyQqen
// SIG // 9rc0JTnQxnpD6tt3cMzU6ULrqMVVAnV4ntgQvgcbypRj
// SIG // +zwPce7UVWfXKfZqvwe85Y+C6vih4nMzPxSSXqG6ZVc3
// SIG // RYWAXNYsxos8aoejBxHL7NlxOCA1JD1UvcxVifR3EJvl
// SIG // maoPrsY8jp/aAE/PfeGzwCD3UjH3IkjVYY1/JgIZfYPM
// SIG // YZMq3bjc4gizp9/hBHdgyPSsNcsx5Wx1UcU3exNqTpVr
// SIG // j0CeyE0+rr1ZxSDWwnitt2RbO9DAFXA0FOXFnqYcx/T/
// SIG // 9stIweKN827OqrSgMIIGcDCCBFigAwIBAgIKYQxSTAAA
// SIG // AAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
// SIG // VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
// SIG // B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
// SIG // b3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3Qg
// SIG // Q2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAw
// SIG // NzA2MjA0MDE3WhcNMjUwNzA2MjA1MDE3WjB+MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQSAyMDEwMIIBIjANBgkqhkiG
// SIG // 9w0BAQEFAAOCAQ8AMIIBCgKCAQEA6Q5kUHlntcTj/QkA
// SIG // TJ6UrPdWaOpE2M/FWE+ppXZ8bUW60zmStKQe+fllguQX
// SIG // 0o/9RJwI6GWTzixVhL99COMuK6hBKxi3oktuSUxrFQfe
// SIG // 0dLCiR5xlM21f0u0rwjYzIjWaxeUOpPOJj/s5v40mFfV
// SIG // HV1J9rIqLtWFu1k/+JC0K4N0yiuzO0bj8EZJwRdmVMkc
// SIG // vR3EVWJXcvhnuSUgNN5dpqWVXqsogM3Vsp7lA7Vj07IU
// SIG // yMHIiiYKWX8H7P8O7YASNUwSpr5SW/Wm2uCLC0h31oVH
// SIG // 1RC5xuiq7otqLQVcYMa0KlucIxxfReMaFB5vN8sZM4Bq
// SIG // iU2jamZjeJPVMM+VHwIDAQABo4IB4zCCAd8wEAYJKwYB
// SIG // BAGCNxUBBAMCAQAwHQYDVR0OBBYEFOb8X3u7IgBY5HJO
// SIG // tfQhdCMy5u+sMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIA
// SIG // QwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/
// SIG // MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjE
// SIG // MFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
// SIG // b0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcB
// SIG // AQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0
// SIG // XzIwMTAtMDYtMjMuY3J0MIGdBgNVHSAEgZUwgZIwgY8G
// SIG // CSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9k
// SIG // ZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUA
// SIG // ZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBt
// SIG // AGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAGnTv
// SIG // V08pe8QWhXi4UNMi/AmdrIKX+DT/KiyXlRLl5L/Pv5PI
// SIG // 4zSp24G43B4AvtI1b6/lf3mVd+UC1PHr2M1OHhthosJa
// SIG // IxrwjKhiUUVnCOM/PB6T+DCFF8g5QKbXDrMhKeWloWmM
// SIG // IpPMdJjnoUdD8lOswA8waX/+0iUgbW9h098H1dlyACxp
// SIG // hnY9UdumOUjJN2FtB91TGcun1mHCv+KDqw/ga5uV1n0o
// SIG // UbCJSlGkmmzItx9KGg5pqdfcwX7RSXCqtq27ckdjF/qm
// SIG // 1qKmhuyoEESbY7ayaYkGx0aGehg/6MUdIdV7+QIjLcVB
// SIG // y78dTMgW77Gcf/wiS0mKbhXjpn92W9FTeZGFndXS2z1z
// SIG // NfM8rlSyUkdqwKoTldKOEdqZZ14yjPs3hdHcdYWch8Za
// SIG // V4XCv90Nj4ybLeu07s8n07VeafqkFgQBpyRnc89NT7be
// SIG // BVaXevfpUk30dwVPhcbYC/GO7UIJ0Q124yNWeCImNr7K
// SIG // sYxuqh3khdpHM2KPpMmRM19xHkCvmGXJIuhCISWKHC1g
// SIG // 2TeJQYkqFg/XYTyUaGBS79ZHmaCAQO4VgXc+nOBTGBpQ
// SIG // HTiVmx5mMxMnORd4hzbOTsNfsvU9R1O24OXbC2E9KteS
// SIG // LM43Wj5AQjGkHxAIwlacvyRdUQKdannSF9PawZSOB3sl
// SIG // cUSrBmrm1MbfI5qWdcUxghXHMIIVwwIBATCBlTB+MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3Nv
// SIG // ZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEwAhMzAAABT+fG
// SIG // YslG9Kl/AAAAAAFPMA0GCWCGSAFlAwQCAQUAoIGyMBkG
// SIG // CSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
// SIG // gjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJ
// SIG // BDEiBCCJpDR7+nzupqvsrn0uzuOV6DoekzetjAkYr6Ab
// SIG // Ih1CsTBGBgorBgEEAYI3AgEMMTgwNqAcgBoAZQByAHIA
// SIG // bwByAC0AaQBuAGYAbwAuAGoAc6EWgBRodHRwOi8vbWlj
// SIG // cm9zb2Z0LmNvbTANBgkqhkiG9w0BAQEFAASCAQBkmEnU
// SIG // OqvJ+A6BZiPrsllDqQ6o8xHDpgcuodv74z/sM2HOmq7a
// SIG // AYZsmlkBxIC+fHNFDrnlmRowdP/Kazn+qaWs+/TMjkQD
// SIG // xVcbm/2qFg+3ZyAH7ugymeAM13D2avBoKPpP3ym+309P
// SIG // oPUKQpph5oMfxctLW7uJxJjtUg+4cSZBZfgrSx75k1dn
// SIG // xoA+97UyGMEe1mEusIBtocjd+H3kEC99kXG4GaYiyYHj
// SIG // qtNmNP4VL1qx0xcF5PE7z0Vq/qvvdrRUUGP6dJpOQonM
// SIG // wv/qzUQvn/nuR27nWLnN8KNJ0/u9IRh/krgQHHbFnFCX
// SIG // qYmBdx3sbs9biBHKGkBNqMgWekgsoYITTTCCE0kGCisG
// SIG // AQQBgjcDAwExghM5MIITNQYJKoZIhvcNAQcCoIITJjCC
// SIG // EyICAQMxDzANBglghkgBZQMEAgEFADCCAT0GCyqGSIb3
// SIG // DQEJEAEEoIIBLASCASgwggEkAgEBBgorBgEEAYRZCgMB
// SIG // MDEwDQYJYIZIAWUDBAIBBQAEIMKQuiWWCIq5CHNyEcm5
// SIG // 3RKWWfqQCa2GEU4k1jZp0V48AgZYP0HLKqcYEzIwMTYx
// SIG // MjI4MDA1ODAzLjU2NVowBwIBAYACAfSggbmkgbYwgbMx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIx
// SIG // JzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVTTjpCQkVDLTMw
// SIG // Q0EtMkRCRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
// SIG // U3RhbXAgU2VydmljZaCCDtAwggZxMIIEWaADAgECAgph
// SIG // CYEqAAAAAAACMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQg
// SIG // Um9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAe
// SIG // Fw0xMDA3MDEyMTM2NTVaFw0yNTA3MDEyMTQ2NTVaMHwx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jv
// SIG // c29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMIIBIjANBgkq
// SIG // hkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqR0NvHcRijog
// SIG // 7PwTl/X6f2mUa3RUENWlCgCChfvtfGhLLF/Fw+Vhwna3
// SIG // PmYrW/AVUycEMR9BGxqVHc4JE458YTBZsTBED/FgiIRU
// SIG // QwzXTbg4CLNC3ZOs1nMwVyaCo0UN0Or1R4HNvyRgMlhg
// SIG // RvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd0WETbijGGvmG
// SIG // gLvfYfxGwScdJGcSchohiq9LZIlQYrFd/XcfPfBXday9
// SIG // ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u5ZrKsajyeioK
// SIG // MfDaTgaRtogINeh4HLDpmc085y9Euqf03GS9pAHBIAmT
// SIG // eM38vMDJRF1eFpwBBU8iTQIDAQABo4IB5jCCAeIwEAYJ
// SIG // KwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNVjOlyKMZDz
// SIG // Q3t8RhvFM2hahW1VMBkGCSsGAQQBgjcUAgQMHgoAUwB1
// SIG // AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fO
// SIG // mhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwu
// SIG // bWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01p
// SIG // Y1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEF
// SIG // BQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cu
// SIG // bWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2Vy
// SIG // QXV0XzIwMTAtMDYtMjMuY3J0MIGgBgNVHSABAf8EgZUw
// SIG // gZIwgY8GCSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYx
// SIG // aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2Nz
// SIG // L0NQUy9kZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIg
// SIG // HQBMAGUAZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABh
// SIG // AHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOC
// SIG // AgEAB+aIUQ3ixuCYP4FxAz2do6Ehb7Prpsz1Mb7PBeKp
// SIG // /vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5vO1rVFcIK1GC
// SIG // RBL7uVOMzPRgEop2zEBAQZvcXBf/XPleFzWYJFZLdO9C
// SIG // EMivv3/Gf/I3fVo/HPKZeUqRUgCvOA8X9S95gWXZqbVr
// SIG // 5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2/QThcJ8ySif9
// SIG // Va8v/rbljjO7Yl+a21dA6fHOmWaQjP9qYn/dxUoLkSbi
// SIG // OewZSnFjnXshbcOco6I8+n99lmqQeKZt0uGc+R38ONiU
// SIG // 9MalCpaGpL2eGq4EQoO4tYCbIjggtSXlZOz39L9+Y1kl
// SIG // D3ouOVd2onGqBooPiRa6YacRy5rYDkeagMXQzafQ732D
// SIG // 8OE7cQnfXXSYIghh2rBQHm+98eEA3+cxB6STOvdlR3jo
// SIG // +KhIq/fecn5ha293qYHLpwmsObvsxsvYgrRyzR30uIUB
// SIG // HoD7G4kqVDmyW9rIDVWZeodzOwjmmC3qjeAzLhIp9cAv
// SIG // VCch98isTtoouLGp25ayp0Kiyc8ZQU3ghvkqmqMRZjDT
// SIG // u3QyS99je/WZii8bxyGvWbWu3EQ8l1Bx16HSxVXjad5X
// SIG // wdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gFLu8NoFA12u8J
// SIG // JxzVs341Hgi62jbb01+P3nSISRIwggTaMIIDwqADAgEC
// SIG // AhMzAAAAoaX9ue6n/J6dAAAAAAChMA0GCSqGSIb3DQEB
// SIG // CwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNo
// SIG // aW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
// SIG // ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMT
// SIG // HU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4X
// SIG // DTE2MDkwNzE3NTY0OFoXDTE4MDkwNzE3NTY0OFowgbMx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIx
// SIG // JzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVTTjpCQkVDLTMw
// SIG // Q0EtMkRCRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUt
// SIG // U3RhbXAgU2VydmljZTCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAJvQAXnmIq96fBt1HvO3Z9Yd3qfe
// SIG // ADTYEvkH6gh7H01NyoWUeqt2ayoHRY1RJ9O2XUalJcZi
// SIG // FHmLT8zmLPXsn016HTQiNiq4w+LsxVDt0/1DwS2/xfD6
// SIG // z8yEkOuiprJtV6CPuWFw1EnOiC+DLjuBSpO1SUrEZcmW
// SIG // heQv+qzi2FcdM/a+EOfCUhsT4PkUNqizm/TbFfD00mt9
// SIG // LbhVzhHpVv+U6SEOu5FXzJwVeXDNjqzT44GZmMMOB9/b
// SIG // c8XPKWP1Puw3GHpz9iR1tfzJN5QwVwfVoEAVTQsWsObV
// SIG // N1082NOrvIeXh3pVc0thQ8HzCYiSvs645YRKPBY3nEHO
// SIG // IJeByAkCAwEAAaOCARswggEXMB0GA1UdDgQWBBSrXLmC
// SIG // vnEq1Y/g6LIpz/ob07f6SDAfBgNVHSMEGDAWgBTVYzpc
// SIG // ijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBH
// SIG // hkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9NaWNUaW1TdGFQQ0FfMjAxMC0wNy0w
// SIG // MS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAC
// SIG // hj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2Nl
// SIG // cnRzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNydDAM
// SIG // BgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMI
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQAjYEy4BVTpZ8m+OTwV
// SIG // MsMTyLMAkH8XM5v9di9dTBKz6LAkov6y9B+oso4uThAH
// SIG // XOj3NIruNB3o4USG+B7b3j0SlIfBmg9D4Jv6FGGrtuoS
// SIG // 46qIatdld+UHW9BQU9HFWtQoLypKUXkfeTkqd8dh5SN/
// SIG // KgXaJp7VQLbiX9w6E3TlOv2q+z8C/Cv5+CBSWyjKXbIg
// SIG // KO4dHvSKhLwXUuqzEnGaLB0bTg140duTDSReF4hJVCH8
// SIG // 2Fwm0OkT4WzKTr8HuDv54ZwvRgxlOpxFsAmnc0wsa7bX
// SIG // FmPg/HI4maYFzTKRQaJe+rvXrQ57hcvzD3c7eDLTT5aR
// SIG // FEgjxoqXl1gd/aVBoYIDeTCCAmECAQEwgeOhgbmkgbYw
// SIG // gbMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
// SIG // dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
// SIG // aWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1P
// SIG // UFIxJzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVTTjpCQkVD
// SIG // LTMwQ0EtMkRCRTElMCMGA1UEAxMcTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgU2VydmljZaIlCgEBMAkGBSsOAwIaBQAD
// SIG // FQCCron1tOY/FlwgvBfkwcNAzTnraKCBwjCBv6SBvDCB
// SIG // uTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9Q
// SIG // UjEnMCUGA1UECxMebkNpcGhlciBOVFMgRVNOOjU3RjYt
// SIG // QzFFMC01NTRDMSswKQYDVQQDEyJNaWNyb3NvZnQgVGlt
// SIG // ZSBTb3VyY2UgTWFzdGVyIENsb2NrMA0GCSqGSIb3DQEB
// SIG // BQUAAgUA3A2FPjAiGA8yMDE2MTIyODAwMjYzOFoYDzIw
// SIG // MTYxMjI5MDAyNjM4WjB3MD0GCisGAQQBhFkKBAExLzAt
// SIG // MAoCBQDcDYU+AgEAMAoCAQACAgEUAgH/MAcCAQACAhr+
// SIG // MAoCBQDcDta+AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwG
// SIG // CisGAQQBhFkKAwGgCjAIAgEAAgMW42ChCjAIAgEAAgMH
// SIG // oSAwDQYJKoZIhvcNAQEFBQADggEBAM1vMo+kY4Qu8XYw
// SIG // b6F0peMOOEdg/hBynJ/vsCHj6UpGf5g8C95FdEjxh8tP
// SIG // peTRaBG6X+ugu0ZbvIn5sL9soToUcw9EqeBJILxTVPm5
// SIG // vmXXgYHQD/FRe/Zx5LDNPBVOFBEyDWnhUv2WgPVBUJdQ
// SIG // xxL1dChRVxkYYv+IS5CJf7ezhkHsceMDQ0q58XcNHo+o
// SIG // jzMqpd8wqj5oegAm8PPHEhhdXMxj9rj+5mDs1X0IjomM
// SIG // x3t2g1Rj040lFPdK7NfqmoVEh4fr8oWSvpQnLfJcqscb
// SIG // lCJ0A/noYjk5FNdSk3zwnUGu5w99Uo33uYCIn1ASW9vA
// SIG // 5sFB8/hzPLKeVHrM53MxggL1MIIC8QIBATCBkzB8MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
// SIG // ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAAAKGl/bnu
// SIG // p/yenQAAAAAAoTANBglghkgBZQMEAgEFAKCCATIwGgYJ
// SIG // KoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3
// SIG // DQEJBDEiBCA4zgoKM6IrCBt7xIF5QaoYcjv0aPFrKJEY
// SIG // 5UKM06PiFDCB4gYLKoZIhvcNAQkQAgwxgdIwgc8wgcww
// SIG // gbEEFIKuifW05j8WXCC8F+TBw0DNOetoMIGYMIGApH4w
// SIG // fDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAACh
// SIG // pf257qf8np0AAAAAAKEwFgQUS1eFn13ukDb/YkJ4npNy
// SIG // t0P/ajEwDQYJKoZIhvcNAQELBQAEggEAYeDPSlbUhx7o
// SIG // 9sxiWPGRAH7198+HQ9uBJSBK7MEXrS4u8DMSu8aIwya3
// SIG // Iqh9CAKKOxYbE/Z0fy1Ga2y5Lio41T+KUMGaVHWdIOnS
// SIG // qBzWrYJR2G8teZdAVIznWeo0fUJLZAcRD6ZXII0MK/Dw
// SIG // YILGZJV8A0xQOuPNdcVc4T0v62MEK5MGmD9zF23XDcRG
// SIG // MT5n9+RZCZD6hu5LQaApB1G67u2QKul8hBBECS52Ucsb
// SIG // TaW0IxS0CBNzuCtMpalY60JImh3GPYVl3qdkHSZ5QSme
// SIG // OSt29TR149O/fN018r/Je2tgq0UHkok3q2oUD+20m7J9
// SIG // 6soJgHui24i8tfzRXL+ioQ==
// SIG // End signature block
